<?php
// Configuration
$file = __DIR__ . '/admin.php';
$backupDir = __DIR__ . '/ADMBackUp';
if(!is_dir($backupDir)) { @mkdir($backupDir, 0777, true); }
$backup = $backupDir . '/admin.php.bak';

// Helper: JSON Response
function json_response($status, $msg) {
    header('Content-Type: application/json');
    echo json_encode(['status' => $status, 'message' => $msg]);
    exit;
}

// Helper: Save with Backup
function save_with_backup($target, $backupPath, $data) {
    // 1. Create Backup
    if (file_exists($target)) { 
        if(!@copy($target, $backupPath)){
            return ['error', 'Failed to create backup permission denied'];
        }
    }

    // 2. Write to Temp File
    $temp = tempnam(sys_get_temp_dir(), 'admin_edit');
    if(!$temp) return ['error', 'Failed to create temp file'];
    
    if (file_put_contents($temp, $data) === false) {
        @unlink($temp);
        return ['error', 'Failed to write data to temp file'];
    }

    // 3. Move Temp to Target (Atomic-like)
    // Try rename first (fastest)
    if (@rename($temp, $target)) {
        @chmod($target, 0644);
        return ['success', 'Saved successfully! Backup created.'];
    }
    
    // Fallback: Unlink + Rename (Windows specific sometimes)
    if (@unlink($target) && @rename($temp, $target)) {
        @chmod($target, 0644);
        return ['success', 'Saved successfully! (Overwrite)'];
    }

    // Fallback: Direct Write (Last resort)
    @unlink($temp);
    if (file_put_contents($target, $data) !== false) {
        return ['success', 'Saved successfully! (Direct Write)'];
    }

    return ['error', 'Failed to save file. Check permissions.'];
}

// Handle POST Requests
if ($_SERVER['REQUEST_METHOD'] === 'POST') {
    // AJAX Save
    if (isset($_POST['ajax_save']) && isset($_POST['content'])) {
        $result = save_with_backup($file, $backup, $_POST['content']);
        json_response($result[0], $result[1]);
    }

    // File Upload (Standard POST)
    if (isset($_POST['upload_php']) && isset($_FILES['php_file']) && $_FILES['php_file']['error'] === UPLOAD_ERR_OK) {
        $name = $_FILES['php_file']['name'] ?? '';
        $ext = strtolower(pathinfo($name, PATHINFO_EXTENSION));
        $tmp = $_FILES['php_file']['tmp_name'];
        $data = file_get_contents($tmp);
        
        if ($ext !== 'php') {
            $uploadMsg = ['error', 'Only .php files are allowed.'];
        } elseif ($data === false || stripos($data, '<?php') === false) {
            $uploadMsg = ['error', 'Upload must be pure PHP.'];
        } else {
            $uploadMsg = save_with_backup($file, $backup, $data);
        }
    }
}

// Load Content
$content = file_exists($file) ? file_get_contents($file) : '';
?>
<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Admin Editor Pro</title>
    <!-- Fonts -->
    <link href="https://fonts.googleapis.com/css2?family=JetBrains+Mono:wght@400;700&family=Inter:wght@400;500;600&display=swap" rel="stylesheet">
    <!-- CodeMirror CSS -->
    <link rel="stylesheet" href="https://cdnjs.cloudflare.com/ajax/libs/codemirror/5.65.2/codemirror.min.css">
    <link rel="stylesheet" href="https://cdnjs.cloudflare.com/ajax/libs/codemirror/5.65.2/theme/dracula.min.css">
    <link rel="stylesheet" href="https://cdnjs.cloudflare.com/ajax/libs/codemirror/5.65.2/addon/scroll/simplescrollbars.min.css">
    <link rel="stylesheet" href="https://cdnjs.cloudflare.com/ajax/libs/codemirror/5.65.2/addon/dialog/dialog.min.css">
    <link rel="stylesheet" href="https://cdnjs.cloudflare.com/ajax/libs/codemirror/5.65.2/addon/search/matchesonscrollbar.css">
    
    <style>
        :root {
            --bg: #121212;
            --panel: #1E1E1E;
            --sidebar: #252526;
            --accent: #E91E63;
            --accent-hover: #C2185B;
            --text: #E0E0E0;
            --text-dim: #A0A0A0;
            --border: #333333;
            --success: #4CAF50;
            --error: #F44336;
            --shadow: 0 4px 6px rgba(0,0,0,0.3);
        }
        
        * { box-sizing: border-box; }
        
        body { 
            margin: 0; padding: 0; 
            background: var(--bg); color: var(--text); 
            font-family: 'Inter', sans-serif; 
            height: 100vh; 
            display: flex; flex-direction: column; 
            overflow: hidden; 
        }
        
        /* Flex Header */
        header {
            flex: 0 0 auto;
            background: var(--panel);
            border-bottom: 1px solid var(--border);
            padding: 10px 20px;
            display: flex;
            align-items: center;
            justify-content: space-between;
            gap: 20px;
            flex-wrap: wrap;
            box-shadow: var(--shadow);
            z-index: 10;
        }
        
        .brand {
            display: flex;
            align-items: center;
            gap: 12px;
            font-weight: 600;
            color: var(--accent);
            text-decoration: none;
            min-width: fit-content;
        }
        
        .brand .icon { font-size: 20px; }
        
        .brand-details {
            display: flex;
            flex-direction: column;
            line-height: 1.2;
        }
        .brand-details .path {
            font-size: 11px;
            color: var(--text-dim);
            font-family: 'JetBrains Mono', monospace;
        }

        /* Toolbar */
        .toolbar {
            display: flex;
            gap: 8px;
            align-items: center;
            flex-wrap: wrap;
            flex: 1;
            justify-content: flex-end;
        }
        
        .group {
            display: flex;
            gap: 8px;
            padding-right: 12px;
            margin-right: 4px;
            border-right: 1px solid var(--border);
            align-items: center;
        }
        .group:last-child { border-right: none; padding-right: 0; margin-right: 0; }

        .btn {
            background: transparent;
            color: var(--text);
            border: 1px solid var(--border);
            padding: 8px 14px;
            border-radius: 6px;
            cursor: pointer;
            font-size: 13px;
            display: flex;
            align-items: center;
            gap: 6px;
            transition: all 0.2s;
            font-weight: 500;
            white-space: nowrap;
        }
        
        .btn:hover { background: rgba(255,255,255,0.05); border-color: var(--text-dim); }
        .btn:active { transform: translateY(1px); }
        .btn:disabled { opacity: 0.5; cursor: not-allowed; }
        
        .btn.primary {
            background: var(--accent);
            border-color: var(--accent);
            color: white;
            box-shadow: 0 2px 4px rgba(233, 30, 99, 0.3);
        }
        .btn.primary:hover { background: var(--accent-hover); border-color: var(--accent-hover); }
        
        .btn.danger { color: var(--error); border-color: rgba(244, 67, 54, 0.3); }
        .btn.danger:hover { background: rgba(244, 67, 54, 0.1); border-color: var(--error); }

        /* File Upload Styling */
        .upload-btn-wrapper {
            position: relative;
            overflow: hidden;
            display: inline-block;
        }
        .upload-btn-wrapper input[type=file] {
            font-size: 100px;
            position: absolute;
            left: 0;
            top: 0;
            opacity: 0;
            cursor: pointer;
        }
        
        /* Editor Area */
        #editor-container { 
            flex: 1; 
            position: relative; 
            overflow: hidden; 
            display: flex;
            flex-direction: column;
        }
        
        #editorForm { height: 100%; flex: 1; }

        /* CodeMirror Overrides */
        .CodeMirror {
            height: 100% !important;
            font-family: 'JetBrains Mono', 'Consolas', monospace;
            font-size: 14px;
            line-height: 1.6;
            background: var(--bg) !important;
        }
        
        .CodeMirror-gutters { background: var(--bg) !important; border-right: 1px solid var(--border); }
        .CodeMirror-linenumber { color: var(--text-dim) !important; }
        
        /* Status Bar */
        .status-bar {
            flex: 0 0 auto;
            background: var(--panel);
            border-top: 1px solid var(--border);
            color: var(--text-dim);
            padding: 5px 15px;
            font-size: 12px;
            display: flex;
            justify-content: space-between;
            align-items: center;
            font-family: 'JetBrains Mono', monospace;
        }
        
        /* Toast */
        .toast {
            position: fixed;
            bottom: 40px;
            right: 30px;
            background: var(--panel);
            color: var(--text);
            padding: 15px 25px;
            border-radius: 8px;
            box-shadow: 0 10px 30px rgba(0,0,0,0.5);
            display: flex;
            align-items: center;
            gap: 15px;
            transform: translateY(150%);
            opacity: 0;
            transition: all 0.3s cubic-bezier(0.68, -0.55, 0.27, 1.55);
            z-index: 1000;
            border: 1px solid var(--border);
            max-width: 400px;
        }
        .toast.show { transform: translateY(0); opacity: 1; }
        .toast.success { border-left: 4px solid var(--success); }
        .toast.error { border-left: 4px solid var(--error); }
        
        .toast-icon { font-size: 20px; }
        .toast-content { display: flex; flex-direction: column; gap: 4px; }
        .toast-title { font-weight: 700; font-size: 14px; }
        .toast-msg { font-size: 13px; color: var(--text-dim); }

        /* Loading Spinner */
        .spinner {
            width: 16px;
            height: 16px;
            border: 2px solid rgba(255,255,255,0.3);
            border-radius: 50%;
            border-top-color: white;
            animation: spin 1s linear infinite;
            display: none;
        }
        @keyframes spin { to { transform: rotate(360deg); } }

        /* Mobile Responsive */
        @media (max-width: 900px) {
            header { padding: 10px; gap: 15px; }
            .brand-details .path { display: none; }
            .toolbar { justify-content: flex-start; width: 100%; overflow-x: auto; padding-bottom: 5px; }
            .group { border-right: none; margin-right: 0; padding-right: 0; }
            .btn .text { display: none; }
            .btn { padding: 8px; }
        }
    </style>
</head>
<body>

    <header>
        <a href="admin.php" class="brand" title="Back to Admin Panel">
            <span class="icon">⬅</span>
            <div class="brand-details">
                <span>Admin Editor</span>
                <span class="path"><?= htmlspecialchars($file) ?></span>
            </div>
        </a>
        
        <div class="toolbar">
            <div class="group">
                <button type="button" class="btn primary" id="saveBtn" onclick="saveFile()" title="Save (Ctrl+S)">
                    <div class="spinner" id="saveSpinner"></div>
                    <span id="saveIcon" class="icon">💾</span> <span id="saveText" class="text">Save</span>
                </button>
                <form id="uploadForm" method="POST" enctype="multipart/form-data" class="upload-btn-wrapper">
                    <button class="btn" type="button" title="Upload PHP File">
                        <span class="icon">⬆</span> <span class="text">Upload</span>
                    </button>
                    <input type="file" name="php_file" accept=".php" onchange="this.form.submit()">
                    <input type="hidden" name="upload_php" value="1">
                </form>
            </div>

            <div class="group">
                <button type="button" class="btn" onclick="undo()" title="Undo (Ctrl+Z)"><span class="icon">↩</span></button>
                <button type="button" class="btn" onclick="redo()" title="Redo (Ctrl+Y)"><span class="icon">↪</span></button>
            </div>
            
            <div class="group">
                <button type="button" class="btn" onclick="toggleSearch()" title="Search (Ctrl+F)"><span class="icon">🔍</span> <span class="text">Search</span></button>
                <button type="button" class="btn" onclick="toggleWrap()" id="wrapBtn" title="Toggle Wrap"><span class="icon">↔</span></button>
                <button type="button" class="btn" onclick="toggleFullscreen()" id="fsBtn" title="Fullscreen (F11)"><span class="icon">⛶</span></button>
            </div>
            
            <div class="group">
                <button type="button" class="btn" onclick="zoom(-1)" title="Zoom Out"><span class="icon">➖</span></button>
                <button type="button" class="btn" onclick="zoom(1)" title="Zoom In"><span class="icon">➕</span></button>
            </div>

             <div class="group">
                <button type="button" class="btn danger" onclick="location.reload()" title="Reload from Disk">
                    <span class="icon">🔄</span> <span class="text">Reset</span>
                </button>
            </div>
        </div>
    </header>

    <div id="editor-container">
        <!-- Hidden Form for standard POST (fallback) -->
        <textarea id="code" style="display:none"><?= htmlspecialchars($content) ?></textarea>
        
        <div class="status-bar">
            <span>PHP Mode</span>
            <span>
                <span id="statusText">Ready</span> | 
                Ln <span id="ln">1</span>, Col <span id="col">1</span>
            </span>
        </div>
    </div>

    <!-- Toast Notification -->
    <div id="toast" class="toast">
        <div class="toast-icon" id="toast-icon"></div>
        <div class="toast-content">
            <div class="toast-title" id="toast-title">Notification</div>
            <div class="toast-msg" id="toast-msg"></div>
        </div>
    </div>

    <!-- Scripts -->
    <script src="https://cdnjs.cloudflare.com/ajax/libs/codemirror/5.65.2/codemirror.min.js"></script>
    <script src="https://cdnjs.cloudflare.com/ajax/libs/codemirror/5.65.2/mode/php/php.min.js"></script>
    <script src="https://cdnjs.cloudflare.com/ajax/libs/codemirror/5.65.2/mode/htmlmixed/htmlmixed.min.js"></script>
    <script src="https://cdnjs.cloudflare.com/ajax/libs/codemirror/5.65.2/mode/xml/xml.min.js"></script>
    <script src="https://cdnjs.cloudflare.com/ajax/libs/codemirror/5.65.2/mode/javascript/javascript.min.js"></script>
    <script src="https://cdnjs.cloudflare.com/ajax/libs/codemirror/5.65.2/mode/css/css.min.js"></script>
    <script src="https://cdnjs.cloudflare.com/ajax/libs/codemirror/5.65.2/mode/clike/clike.min.js"></script>
    <script src="https://cdnjs.cloudflare.com/ajax/libs/codemirror/5.65.2/addon/edit/matchbrackets.min.js"></script>
    <script src="https://cdnjs.cloudflare.com/ajax/libs/codemirror/5.65.2/addon/edit/closebrackets.min.js"></script>
    <script src="https://cdnjs.cloudflare.com/ajax/libs/codemirror/5.65.2/addon/selection/active-line.min.js"></script>
    <script src="https://cdnjs.cloudflare.com/ajax/libs/codemirror/5.65.2/addon/scroll/simplescrollbars.min.js"></script>
    <script src="https://cdnjs.cloudflare.com/ajax/libs/codemirror/5.65.2/addon/dialog/dialog.min.js"></script>
    <script src="https://cdnjs.cloudflare.com/ajax/libs/codemirror/5.65.2/addon/search/searchcursor.min.js"></script>
    <script src="https://cdnjs.cloudflare.com/ajax/libs/codemirror/5.65.2/addon/search/search.min.js"></script>
    <script src="https://cdnjs.cloudflare.com/ajax/libs/codemirror/5.65.2/addon/search/jump-to-line.min.js"></script>

    <script>
        // Initialize CodeMirror
        var editor = CodeMirror.fromTextArea(document.getElementById("code"), {
            lineNumbers: true,
            mode: "application/x-httpd-php",
            theme: "dracula",
            indentUnit: 4,
            indentWithTabs: true,
            matchBrackets: true,
            autoCloseBrackets: true,
            styleActiveLine: true,
            scrollbarStyle: "simple",
            extraKeys: {
                "Ctrl-S": function(cm) { saveFile(); },
                "Cmd-S": function(cm) { saveFile(); },
                "Ctrl-F": "findPersistent"
            }
        });

        // Cursor Position Tracker
        editor.on("cursorActivity", function(cm) {
            var pos = cm.getCursor();
            document.getElementById('ln').innerText = pos.line + 1;
            document.getElementById('col').innerText = pos.ch + 1;
        });
        
        // Change tracker
        var isDirty = false;
        editor.on("change", function() {
            if(!isDirty) {
                isDirty = true;
                document.title = "* Admin Editor Pro";
                document.getElementById('statusText').innerText = "Unsaved";
                document.getElementById('statusText').style.color = "#FF9800";
            }
        });

        // AJAX Save Function
        async function saveFile() {
            const btn = document.getElementById('saveBtn');
            const spinner = document.getElementById('saveSpinner');
            const icon = document.getElementById('saveIcon');
            const text = document.getElementById('saveText');
            const content = editor.getValue();

            // UI Loading State
            btn.disabled = true;
            spinner.style.display = 'block';
            icon.style.display = 'none';
            text.innerText = 'Saving...';
            document.getElementById('statusText').innerText = "Saving...";

            const formData = new FormData();
            formData.append('ajax_save', '1');
            formData.append('content', content);

            try {
                const response = await fetch(window.location.href, {
                    method: 'POST',
                    body: formData
                });

                if (!response.ok) throw new Error('Network response was not ok');
                
                const data = await response.json();
                
                if (data.status === 'success') {
                    showToast(data.message, 'success');
                    isDirty = false;
                    document.title = "Admin Editor Pro";
                    document.getElementById('statusText').innerText = "Saved";
                    document.getElementById('statusText').style.color = "#4CAF50";
                } else {
                    showToast(data.message || 'Unknown error', 'error');
                    document.getElementById('statusText').innerText = "Error";
                    document.getElementById('statusText').style.color = "#F44336";
                }

            } catch (error) {
                console.error('Error:', error);
                showToast('Connection failed or invalid response', 'error');
                document.getElementById('statusText').innerText = "Conn Error";
            } finally {
                // UI Reset
                btn.disabled = false;
                spinner.style.display = 'none';
                icon.style.display = 'inline';
                text.innerText = 'Save';
            }
        }

        function undo() { editor.undo(); }
        function redo() { editor.redo(); }
        
        function toggleSearch() {
            editor.execCommand("findPersistent");
        }

        var fontSize = 14;
        function zoom(delta) {
            fontSize = Math.max(10, Math.min(24, fontSize + (delta > 0 ? 1 : -1)));
            editor.getWrapperElement().style.fontSize = fontSize + 'px';
            editor.refresh();
        }

        var isWrapped = false;
        function toggleWrap() {
            isWrapped = !isWrapped;
            editor.setOption('lineWrapping', isWrapped);
            var b = document.getElementById('wrapBtn');
            b.classList.toggle('primary');
            showToast('Line wrapping ' + (isWrapped ? 'enabled' : 'disabled'), 'success');
        }

        function toggleFullscreen() {
            var el = document.documentElement;
            var b = document.getElementById('fsBtn');
            if (!document.fullscreenElement) {
                if (el.requestFullscreen) { el.requestFullscreen(); }
            } else {
                if (document.exitFullscreen) { document.exitFullscreen(); }
            }
        }

        function showToast(msg, type) {
            const toast = document.getElementById('toast');
            const icon = document.getElementById('toast-icon');
            const title = document.getElementById('toast-title');
            const text = document.getElementById('toast-msg');
            
            toast.className = 'toast show ' + type;
            icon.innerText = type === 'success' ? '✅' : '❌';
            title.innerText = type === 'success' ? 'Success' : 'Error';
            text.innerText = msg;
            
            // Clear previous timeout if exists
            if (toast.timeoutId) clearTimeout(toast.timeoutId);
            
            toast.timeoutId = setTimeout(() => {
                toast.className = 'toast';
            }, 3000);
        }

        // Prevent accidental leave
        window.onbeforeunload = function() {
            if (isDirty) return "You have unsaved changes. Are you sure?";
        };

        // Handle PHP Upload Messages (from page reload)
        <?php if(isset($uploadMsg)): ?>
            showToast(<?= json_encode($uploadMsg[1]) ?>, <?= json_encode($uploadMsg[0]) ?>);
        <?php endif; ?>
    </script>
</body>
</html>
