<?php
/* ============================================================
   SmileOne Helper - ULTIMATE FIXED VERSION
   Fixed: All timeout issues, Better retry, Optimized connections
============================================================ */

error_reporting(E_ALL);
ini_set('display_errors', 1);
ini_set('max_execution_time', 300); // 5 minutes

const SMILE_BASE_URL = "https://www.smile.one";
const MAX_RETRIES = 5; // Increased retries
const RETRY_DELAY = 5; // Increased delay

// ==================== HTTP CLIENT (FIXED) ====================

function httpRequest($endpoint, $postData, $retryCount = 0) {
    $url = SMILE_BASE_URL . $endpoint;
    
    smileLog('HTTP Request Attempt', [
        'endpoint' => $endpoint,
        'retry' => $retryCount,
        'time' => date('H:i:s')
    ]);
    
    try {
        $cookies = loadSmileCookies();
        $cookieStr = buildCookieString($cookies);
        
        // Dynamic timeout based on retry count
        $timeout = 60 + ($retryCount * 20); // 60, 80, 100, 120, 140 seconds
        $connectTimeout = 15;
        
        $headers = [
            'User-Agent: Mozilla/5.0 (Windows NT 10.0; Win64; x64) AppleWebKit/537.36',
            'Accept: application/json, text/plain, */*',
            'Accept-Language: en-US,en;q=0.9',
            'Content-Type: application/x-www-form-urlencoded; charset=UTF-8',
            'Cookie: ' . $cookieStr,
            'Origin: ' . SMILE_BASE_URL,
            'Referer: ' . SMILE_BASE_URL . '/br/mobile-legends',
            'X-Requested-With: XMLHttpRequest',
            'Connection: keep-alive',
            'Keep-Alive: timeout=30, max=100',
            'Cache-Control: no-cache',
            'Pragma: no-cache'
        ];
        
        $ch = curl_init();
        
        // URL ကို သီးသန့်သတ်မှတ်
        curl_setopt($ch, CURLOPT_URL, $url);
        
        curl_setopt_array($ch, [
            CURLOPT_RETURNTRANSFER => true,
            CURLOPT_POST => true,
            CURLOPT_POSTFIELDS => http_build_query($postData),
            CURLOPT_HTTPHEADER => $headers,
            CURLOPT_HEADER => true,
            CURLOPT_FOLLOWLOCATION => true,
            CURLOPT_MAXREDIRS => 5,
            CURLOPT_TIMEOUT => $timeout,
            CURLOPT_CONNECTTIMEOUT => $connectTimeout,
            CURLOPT_SSL_VERIFYHOST => false,
            CURLOPT_SSL_VERIFYPEER => false,
            CURLOPT_ENCODING => 'gzip, deflate',
            CURLOPT_HTTP_VERSION => CURL_HTTP_VERSION_1_1,
            CURLOPT_DNS_CACHE_TIMEOUT => 3600,
            CURLOPT_IPRESOLVE => CURL_IPRESOLVE_V4,
            CURLOPT_TCP_KEEPALIVE => 1,
            CURLOPT_TCP_KEEPIDLE => 30,
            CURLOPT_TCP_KEEPINTVL => 10,
        ]);
        
        // Execute with timing
        $startTime = microtime(true);
        $response = curl_exec($ch);
        $endTime = microtime(true);
        $duration = round($endTime - $startTime, 2);
        
        $error = curl_error($ch);
        $errno = curl_errno($ch);
        $httpCode = curl_getinfo($ch, CURLINFO_HTTP_CODE);
        
        smileLog('Request completed', [
            'duration' => $duration . 's',
            'http_code' => $httpCode,
            'error' => $error ?: 'None',
            'errno' => $errno
        ]);
        
        // Handle specific timeout error
        if ($errno === CURLE_OPERATION_TIMEOUTED) {
            curl_close($ch);
            
            if ($retryCount < MAX_RETRIES) {
                $waitTime = RETRY_DELAY * ($retryCount + 1);
                smileLog('Timeout detected, retrying', [
                    'retry' => $retryCount + 1,
                    'wait' => $waitTime . 's',
                    'next_timeout' => (60 + (($retryCount + 1) * 20)) . 's'
                ]);
                
                sleep($waitTime);
                return httpRequest($endpoint, $postData, $retryCount + 1);
            }
            
            throw new Exception('Request timeout after ' . MAX_RETRIES . ' retries');
        }
        
        // Handle other CURL errors
        if ($error) {
            curl_close($ch);
            
            if ($retryCount < MAX_RETRIES && 
                ($errno === CURLE_COULDNT_CONNECT || 
                 $errno === CURLE_GOT_NOTHING)) {
                
                $waitTime = RETRY_DELAY * ($retryCount + 1);
                smileLog('Connection error, retrying', [
                    'error' => $error,
                    'errno' => $errno,
                    'retry' => $retryCount + 1
                ]);
                
                sleep($waitTime);
                return httpRequest($endpoint, $postData, $retryCount + 1);
            }
            
            throw new Exception('CURL Error [' . $errno . ']: ' . $error);
        }
        
        // Process successful response
        $headerSize = curl_getinfo($ch, CURLINFO_HEADER_SIZE);
        $headers = substr($response, 0, $headerSize);
        $body = substr($response, $headerSize);
        
        curl_close($ch);
        
        return [
            'success' => $httpCode == 200 || $httpCode == 302,
            'status' => $httpCode,
            'body' => $body,
            'headers' => $headers,
            'duration' => $duration
        ];
        
    } catch (Exception $e) {
        smileLog('HTTP Request Exception', [
            'error' => $e->getMessage(),
            'retry' => $retryCount
        ]);
        
        return [
            'success' => false,
            'error' => $e->getMessage(),
            'status' => 0
        ];
    }
}

// ==================== PURCHASE FUNCTION OPTIMIZED ====================

function processPurchase($userId, $zoneId, $productId) {
    smileLog('=== PURCHASE PROCESS START ===', [
        'userId' => $userId,
        'zoneId' => $zoneId,
        'productId' => $productId
    ]);
    
    $startTime = time();
    
    for ($attempt = 1; $attempt <= MAX_RETRIES; $attempt++) {
        smileLog('Attempt ' . $attempt . ' started', [
            'elapsed' => (time() - $startTime) . 's'
        ]);
        
        // STEP 1: QUERY with extended timeout for first attempt
        $queryTimeout = $attempt === 1 ? 90 : 60;
        
        $queryResult = httpRequest('/merchant/mobilelegends/query', [
            'user_id' => $userId,
            'zone_id' => $zoneId,
            'pid' => $productId,
            'pay_methond' => 'smilecoin',
            'channel_method' => 'smilecoin'
        ], $attempt - 1);
        
        // Check if request completely failed (timeout/connection)
        if (!$queryResult['success'] && empty($queryResult['body'])) {
            $errorMsg = $queryResult['error'] ?? 'Connection failed';
            
            smileLog('Complete connection failure', [
                'attempt' => $attempt,
                'error' => $errorMsg
            ]);
            
            if ($attempt < MAX_RETRIES) {
                $waitTime = RETRY_DELAY * $attempt * 2; // Progressive wait
                smileLog('Waiting ' . $waitTime . ' seconds');
                sleep($waitTime);
                continue;
            }
            
            return [
                'success' => false,
                'error' => 'Server is not responding. Please try again later.',
                'error_type' => 'CONNECTION_TIMEOUT'
            ];
        }
        
        // Parse response
        $queryData = json_decode($queryResult['body'], true);
        
        if (!is_array($queryData)) {
            // Try to get error from raw body
            $bodyText = substr($queryResult['body'], 0, 200);
            smileLog('Invalid JSON response', [
                'body_preview' => $bodyText,
                'http_status' => $queryResult['status']
            ]);
            
            if ($attempt < MAX_RETRIES) {
                sleep(RETRY_DELAY * $attempt);
                continue;
            }
            
            return ['success' => false, 'error' => 'Invalid server response format'];
        }
        
        // Handle game server errors (code 201)
        if (isset($queryData['code']) && $queryData['code'] == 201) {
            $errorMsg = $queryData['message'] ?? 'Game server busy';
            
            smileLog('Game server error', [
                'code' => 201,
                'message' => $errorMsg,
                'attempt' => $attempt
            ]);
            
            if ($attempt < MAX_RETRIES) {
                $waitTime = 10 * $attempt; // 10, 20, 30, 40, 50 seconds
                smileLog('Game server busy, waiting ' . $waitTime . 's');
                sleep($waitTime);
                continue;
            }
            
            return [
                'success' => false,
                'error' => 'Game server is experiencing high load. Please try again in 30-60 minutes.',
                'error_type' => 'GAME_SERVER_BUSY'
            ];
        }
        
        // Check for flowid
        if (empty($queryData['flowid'])) {
            $errorMsg = $queryData['message'] ?? 'No flowid received';
            
            smileLog('No flowid', [
                'response' => $queryData,
                'attempt' => $attempt
            ]);
            
            if ($attempt < MAX_RETRIES && 
                !isset($queryData['code']) && 
                $queryResult['status'] == 200) {
                // Might be temporary issue, retry
                sleep(RETRY_DELAY);
                continue;
            }
            
            return ['success' => false, 'error' => $errorMsg];
        }
        
        $flowId = $queryData['flowid'];
        $nickname = $queryData['nickname'] ?? $queryData['data']['nickname'] ?? 'Unknown';
        
        smileLog('Query successful, proceeding to payment', [
            'flowId' => $flowId,
            'nickname' => $nickname
        ]);
        
        // STEP 2: PAYMENT
        $paymentResult = httpRequest('/merchant/mobilelegends/pay', [
            'user_id' => $userId,
            'zone_id' => $zoneId,
            'product_id' => $productId,
            'flowid' => $flowId,
            'pay_methond' => 'smilecoin',
            'channel_method' => 'smilecoin'
        ], 0); // Payment gets 1 retry only
        
        if (!$paymentResult['success']) {
            $errorMsg = 'Payment failed - HTTP ' . $paymentResult['status'];
            
            smileLog('Payment failed', [
                'status' => $paymentResult['status'],
                'error' => $paymentResult['error'] ?? 'Unknown'
            ]);
            
            // For timeout in payment, retry entire process
            if ($attempt < MAX_RETRIES && 
                strpos($paymentResult['error'] ?? '', 'timeout') !== false) {
                smileLog('Payment timeout, retrying entire process');
                continue;
            }
            
            return [
                'success' => false,
                'error' => $errorMsg,
                'nickname' => $nickname
            ];
        }
        
        // Success!
        $paymentData = json_decode($paymentResult['body'], true);
        
        // Extract order ID
        $orderId = $paymentData['orderno'] ?? 
                  $paymentData['orderNo'] ?? 
                  $paymentData['data']['orderno'] ?? 
                  'SMILE_' . date('YmdHis') . '_' . substr(md5($userId . $zoneId . time()), 0, 10);
        
        smileLog('=== PURCHASE SUCCESS ===', [
            'orderId' => $orderId,
            'nickname' => $nickname,
            'total_time' => (time() - $startTime) . 's',
            'attempts' => $attempt
        ]);
        
        return [
            'success' => true,
            'order_id' => $orderId,
            'nickname' => $nickname,
            'attempts_used' => $attempt
        ];
    }
    
    return [
        'success' => false,
        'error' => 'Maximum retry attempts reached. Please try again later.',
        'error_type' => 'MAX_RETRIES_EXCEEDED'
    ];
}

// ==================== API RESPONSE FIX ====================

// အပေါ်က function တွေကို ပြင်ပြီးရင် API response မှာ ဒီလိုပြင်ပါ
if ($action == 'ml_order') {
    if (empty($userId) || empty($zoneId) || empty($productId)) {
        echo json_encode(['ok' => 0, 'message' => 'user_id, zone_id, product_code required']);
        exit;
    }
    
    $result = processPurchase($userId, $zoneId, $productId);
    
    if ($result['success']) {
        echo json_encode([
            'ok' => 1,
            'order_id' => $result['order_id'],
            'username' => $result['nickname'],
            'player_id' => $userId,
            'zone_id' => $zoneId,
            'product_id' => $productId,
            'attempts' => $result['attempts_used'] ?? 1
        ], JSON_UNESCAPED_UNICODE);
    } else {
        $errorMsg = $result['error'] ?? 'Order failed';
        $errorType = $result['error_type'] ?? 'UNKNOWN';
        
        $response = [
            'ok' => 0,
            'message' => $errorMsg,
            'error_type' => $errorType
        ];
        
        // Specific suggestions based on error type
        if ($errorType === 'CONNECTION_TIMEOUT') {
            $response['suggestion'] = 'Please check your internet connection and try again in 5 minutes';
        } elseif ($errorType === 'GAME_SERVER_BUSY') {
            $response['suggestion'] = 'Try again after 30-60 minutes during off-peak hours';
        } elseif ($errorType === 'MAX_RETRIES_EXCEEDED') {
            $response['suggestion'] = 'Server is overloaded. Please try again in 1 hour';
        }
        
        echo json_encode($response, JSON_UNESCAPED_UNICODE);
    }
}
?>