<?php

const SMILE_BASE_URL = 'https://www.smile.one';

function getSmileCookies(): string {
    $jsonPath = __DIR__ . '/cookies.json';
    if (!file_exists($jsonPath)) {
        throw new Exception("cookies.json not found.");
    }

    $cookies = json_decode(file_get_contents($jsonPath), true);
    if (!is_array($cookies)) {
        throw new Exception("Invalid cookies.json format.");
    }

    $cookieString = "";
    foreach ($cookies as $cookie) {
        $cookieString .= $cookie['name'] . '=' . $cookie['value'] . '; ';
    }
    return rtrim($cookieString, '; ');
}

function buildSmileHeaders(array $overrides = [], ?string $cookieString = null): array {
    $headers = [
        'User-Agent' => 'Mozilla/5.0 (Windows NT 10.0; Win64; x64) AppleWebKit/537.36 (KHTML, like Gecko) Chrome/91.0.4472.124 Safari/537.36',
        'Accept' => 'application/json',
        'Cookie' => $cookieString ?? getSmileCookies(),
    ];

    foreach ($overrides as $name => $value) {
        if ($value === null) {
            unset($headers[$name]);
            continue;
        }
        $headers[$name] = $value;
    }

    $formatted = [];
    foreach ($headers as $name => $value) {
        $formatted[] = $name . ': ' . $value;
    }

    return $formatted;
}

function parseSmileHeaderLines(array $lines): array {
    $headers = [];

    foreach ($lines as $line) {
        $line = trim($line);
        if ($line === '' || stripos($line, 'HTTP/') === 0) {
            continue;
        }

        $delimiterPos = strpos($line, ':');
        if ($delimiterPos === false) {
            continue;
        }

        $name = trim(substr($line, 0, $delimiterPos));
        $value = trim(substr($line, $delimiterPos + 1));
        if ($name !== '') {
            $headers[$name] = $value;
        }
    }

    return $headers;
}

function parseSmileHeaderString(string $raw): array {
    $raw = trim($raw);
    if ($raw === '') {
        return [];
    }

    $sections = preg_split("/\r\n\r\n|\n\n/", $raw);
    $lastSection = $sections ? end($sections) : '';
    $lines = preg_split("/\r\n|\n/", $lastSection) ?: [];

    return parseSmileHeaderLines($lines);
}

function sendSmilePost(string $url, array $data, array $options = []): array {
    $isForm = (bool)($options['form'] ?? false);
    $timeout = (int)($options['timeout'] ?? 10);
    $cookies = $options['cookies'] ?? null;
    $headerOverrides = $options['headers'] ?? [];

    if (!isset($headerOverrides['Content-Type'])) {
        $headerOverrides['Content-Type'] = $isForm ? 'application/x-www-form-urlencoded' : 'application/json';
    }

    $postFields = $isForm ? http_build_query($data) : json_encode($data);
    if ($postFields === false) {
        throw new Exception('Failed to encode request payload.');
    }

    $headers = buildSmileHeaders($headerOverrides, $cookies);

    if (function_exists('curl_init')) {
        $ch = curl_init($url);
        curl_setopt_array($ch, [
            CURLOPT_RETURNTRANSFER => true,
            CURLOPT_POST => true,
            CURLOPT_POSTFIELDS => $postFields,
            CURLOPT_HTTPHEADER => $headers,
            CURLOPT_FOLLOWLOCATION => false,
            CURLOPT_TIMEOUT => $timeout,
            CURLOPT_HEADER => true,
        ]);

        $rawResponse = curl_exec($ch);
        $httpCode = curl_getinfo($ch, CURLINFO_HTTP_CODE);
        $headerSize = curl_getinfo($ch, CURLINFO_HEADER_SIZE) ?: 0;
        $err = curl_error($ch);

        $rawHeaders = '';
        $body = null;
        if ($rawResponse !== false) {
            $rawHeaders = substr($rawResponse, 0, $headerSize);
            $body = substr($rawResponse, $headerSize);
        }
        curl_close($ch);

        $parsedHeaders = parseSmileHeaderString($rawHeaders);
        $redirectUrl = $parsedHeaders['Location'] ?? null;

        if ($body === null && !$err && $rawResponse === false) {
            $err = 'Unknown cURL error';
        }

        return [
            'body' => $body,
            'status' => $httpCode,
            'redirect' => $redirectUrl,
            'error' => $err !== '' ? $err : null,
            'headers' => $parsedHeaders,
        ];
    }

    $context = stream_context_create([
        'http' => [
            'method' => 'POST',
            'header' => implode("\r\n", $headers),
            'content' => $postFields,
            'ignore_errors' => true,
            'timeout' => $timeout,
        ],
    ]);

    $response = @file_get_contents($url, false, $context);
    $httpCode = 0;
    $redirectUrl = null;
    $err = null;
    $parsedHeaders = [];

    if (isset($http_response_header) && is_array($http_response_header)) {
        $parsedHeaders = parseSmileHeaderLines($http_response_header);
        foreach ($http_response_header as $headerLine) {
            if (preg_match('/^HTTP\/\S+\s+(\d{3})/', $headerLine, $statusMatch)) {
                $httpCode = (int)$statusMatch[1];
            } elseif (stripos($headerLine, 'Location:') === 0) {
                $redirectUrl = trim(substr($headerLine, 9));
            }
        }
    }

    if ($response === false) {
        $lastError = error_get_last();
        $err = $lastError['message'] ?? 'HTTP request failed';
    }

    return [
        'body' => $response !== false ? $response : null,
        'status' => $httpCode,
        'redirect' => $redirectUrl,
        'error' => $err,
        'headers' => $parsedHeaders,
    ];
}

function getSmileBalance(): array {
    try {
        $cookies = getSmileCookies();
        $url = "https://www.smile.one/customer/order";
        $headers = [
            "User-Agent: Mozilla/5.0",
            "Cookie: $cookies",
        ];

        if (function_exists('curl_init')) {
            $ch = curl_init($url);
            curl_setopt_array($ch, [
                CURLOPT_RETURNTRANSFER => true,
                CURLOPT_HTTPHEADER => $headers,
                CURLOPT_TIMEOUT => 10,
            ]);

            $html = curl_exec($ch);
            if ($html === false) {
                throw new Exception(curl_error($ch) ?: 'Unknown cURL error');
            }
            curl_close($ch);
        } else {
            $context = stream_context_create([
                'http' => [
                    'method' => 'GET',
                    'header' => implode("\r\n", $headers),
                    'ignore_errors' => true,
                    'timeout' => 10,
                ],
            ]);

            $html = @file_get_contents($url, false, $context);
            if ($html === false) {
                $lastError = error_get_last();
                throw new Exception($lastError['message'] ?? 'Unable to fetch balance page');
            }
        }

        if (preg_match('/<div class="balance-coins">.*?<p>(.*?)<\/p>.*?<p>(.*?)<\/p>/s', $html, $matches)) {
            return ['success' => true, 'balance' => trim(strip_tags($matches[2]))];
        } else {
            return ['success' => false, 'message' => 'Balance not found'];
        }
    } catch (Exception $e) {
        return ['success' => false, 'message' => $e->getMessage()];
    }
}

function checkSmileUserRole(string $user_id, string $zone_id): array {
    try {
        $cookies = getSmileCookies();
        $postData = [
            'user_id' => $user_id,
            'zone_id' => $zone_id,
            'pid' => '25',
            'checkrole' => '1',
            'pay_methond' => '',
            'channel_method' => '',
        ];

        $response = sendSmilePost(SMILE_BASE_URL . "/merchant/mobilelegends/checkrole", $postData, [
            'cookies' => $cookies,
        ]);

        if ($response['error']) {
            throw new Exception($response['error']);
        }

        if ($response['status'] >= 500) {
            throw new Exception("Remote service error ({$response['status']}).");
        }

        $json = json_decode($response['body'] ?? '', true);
        if (!is_array($json)) {
            throw new Exception('Invalid response payload.');
        }

        if (($json['code'] ?? null) === 200 && isset($json['username'])) {
            return ['success' => true, 'username' => $json['username']];
        } else {
            return ['success' => false, 'message' => $json['info'] ?? 'Checkrole failed'];
        }
    } catch (Exception $e) {
        return ['success' => false, 'message' => $e->getMessage()];
    }
}

function performSmilePurchase(string $user_id, string $zone_id, string $product_id, string $cookies): array {
    error_log("Purchase request: userid={$user_id}, zoneid={$zone_id}, productid={$product_id}");

    try {
        $checkPayload = [
            'user_id' => $user_id,
            'zone_id' => $zone_id,
            'pid' => $product_id,
            'checkrole' => 1,
            'pay_methond' => 'null',
            'channel_method' => 'null',
        ];

        $checkResponse = sendSmilePost(SMILE_BASE_URL . '/merchant/mobilelegends/checkrole', $checkPayload, [
            'cookies' => $cookies,
        ]);

        if ($checkResponse['error']) {
            throw new Exception($checkResponse['error']);
        }

        if ($checkResponse['status'] >= 500) {
            throw new Exception("Remote service error ({$checkResponse['status']}).");
        }

        $checkJson = json_decode($checkResponse['body'] ?? '', true);
        if (!is_array($checkJson)) {
            throw new Exception('Invalid response payload');
        }

        if (($checkJson['code'] ?? null) !== 200) {
            $info = $checkJson['info'] ?? 'Unknown error';
            throw new Exception("Checkrole failed: {$info}");
        }
    } catch (Exception $error) {
        return ['success' => false, 'error' => 'Checkrole error: ' . $error->getMessage()];
    }

    $retries = 0;
    $lastError = null;

    while ($retries < 3) {
        try {
            $queryPayload = [
                'user_id' => $user_id,
                'zone_id' => $zone_id,
                'pid' => $product_id,
                'checkrole' => null,
                'pay_methond' => 'smilecoin',
                'channel_method' => 'smilecoin',
            ];

            $queryResponse = sendSmilePost(SMILE_BASE_URL . '/merchant/mobilelegends/query', $queryPayload, [
                'cookies' => $cookies,
            ]);

            if ($queryResponse['error']) {
                throw new Exception($queryResponse['error']);
            }

            if ($queryResponse['status'] >= 500) {
                throw new Exception("Remote service error ({$queryResponse['status']}).");
            }

            $queryJson = json_decode($queryResponse['body'] ?? '', true);
            if (!is_array($queryJson)) {
                throw new Exception('Invalid response payload');
            }

            if (($queryJson['code'] ?? null) !== 200) {
                $info = $queryJson['info'] ?? 'Unknown error';
                throw new Exception("Query failed: {$info}");
            }

            $flowId = $queryJson['flowid'] ?? null;
            if (!is_string($flowId) || $flowId === '') {
                throw new Exception('Invalid flow ID received');
            }

            $payPayload = [
                'user_id' => $user_id,
                'zone_id' => $zone_id,
                'product_id' => $product_id,
                'flowid' => $flowId,
                'pay_methond' => 'smilecoin',
                'channel_method' => 'smilecoin',
                'email' => null,
                'coupon_id' => null,
            ];

            $payResponse = sendSmilePost(SMILE_BASE_URL . '/merchant/mobilelegends/pay', $payPayload, [
                'cookies' => $cookies,
            ]);

            if ($payResponse['error']) {
                throw new Exception($payResponse['error']);
            }

            $status = $payResponse['status'];
            if ($status < 300 || $status >= 400) {
                throw new Exception("Unexpected status code {$status}");
            }

            $redirectUrl = $payResponse['redirect'] ?? null;
            if ($redirectUrl === SMILE_BASE_URL . '/message/success') {
                error_log("Purchase success: userid={$user_id}, zoneid={$zone_id}, pid={$product_id}");
                return ['success' => true];
            }

            throw new Exception('Payment failed or unexpected redirect');
        } catch (Exception $error) {
            $lastError = $error;
            $retries++;
            error_log("Purchase attempt {$retries} failed: " . $error->getMessage());
            if ($retries < 3) {
                usleep(2000000);
            }
        }
    }

    return ['success' => false, 'error' => $lastError ? $lastError->getMessage() : 'Unknown error'];
}

function purchaseSmileProduct(string $user_id, string $zone_id, string|array $product_ids): array {
    try {
        $cookies = getSmileCookies();
    } catch (Exception $e) {
        return ['success' => false, 'error' => 'Service unavailable. Please try again later'];
    }

    $productList = is_array($product_ids) ? array_values($product_ids) : [$product_ids];
    $results = [];

    foreach ($productList as $index => $id) {
        $productId = (string)$id;
        $result = performSmilePurchase($user_id, $zone_id, $productId, $cookies);
        $results[] = $result + [
            'product_id' => $productId,
            'sequence' => $index,
        ];

        if (!is_array($product_ids)) {
            return $result;
        }
    }

    $allSuccess = !empty($results) && array_reduce($results, function (bool $carry, array $item) {
        return $carry && ($item['success'] ?? false);
    }, true);

    return [
        'success' => $allSuccess,
        'results' => $results,
    ];
}
