<?php
/* ============================================================
   SmileOne Helper - GAME SERVER ERROR FIXED VERSION
   Fixed: Game server error, Retry mechanism, Better timeout
============================================================ */

error_reporting(E_ALL);
ini_set('display_errors', 1);
ini_set('max_execution_time', 300); // 5 minutes

const SMILE_BASE_URL = "https://www.smile.one";
const MAX_RETRIES = 5;
const RETRY_DELAY = 5; // seconds between retries

// ==================== LOGGING ====================

function smileLog($message, $data = []) {
    $log = date('Y-m-d H:i:s') . ' | ' . $message;
    if (!empty($data)) {
        $log .= ' | ' . json_encode($data, JSON_UNESCAPED_UNICODE);
    }
    $log .= PHP_EOL;
    file_put_contents(__DIR__ . '/smile_debug.log', $log, FILE_APPEND);
    return $log;
}

// ==================== COOKIE HANDLER ====================

function loadSmileCookies() {
    $file = __DIR__ . '/cookies.json';
    
    if (!file_exists($file)) {
        throw new Exception('cookies.json not found');
    }
    
    $content = file_get_contents($file);
    $cookies = json_decode($content, true);
    
    if (json_last_error() !== JSON_ERROR_NONE) {
        throw new Exception('Invalid cookies.json: ' . json_last_error_msg());
    }
    
    if (!is_array($cookies)) {
        throw new Exception('cookies.json should be an array');
    }
    
    smileLog('Cookies loaded', ['count' => count($cookies)]);
    
    // Check essential cookies
    $essential = ['PHPSESSID', '_csrf'];
    $found = [];
    
    foreach ($cookies as $cookie) {
        $name = $cookie['name'] ?? '';
        if (in_array($name, $essential)) {
            $found[] = $name;
        }
    }
    
    if (count($found) < 2) {
        smileLog('Missing essential cookies', [
            'found' => $found,
            'required' => $essential
        ]);
    }
    
    return $cookies;
}

function buildCookieString($cookies) {
    $cookieStr = '';
    
    foreach ($cookies as $cookie) {
        if (isset($cookie['name'], $cookie['value'])) {
            $name = $cookie['name'];
            $value = $cookie['value'];
            
            // Decode URL encoding
            $decoded = urldecode($value);
            
            // Handle serialized PHP data (common in SmileOne)
            if (strpos($decoded, 'a:2:{') === 0) {
                $unserialized = @unserialize($decoded);
                if ($unserialized !== false && isset($unserialized[1])) {
                    $value = $unserialized[1];
                } else {
                    $value = $decoded;
                }
            } else {
                $value = $decoded;
            }
            
            $cookieStr .= $name . '=' . urlencode($value) . '; ';
        }
    }
    
    return trim($cookieStr);
}

// ==================== HTTP CLIENT (IMPROVED) ====================

function httpRequest($endpoint, $postData, $retryCount = 0) {
    $url = SMILE_BASE_URL . $endpoint;
    
    smileLog('HTTP Request', [
        'endpoint' => $endpoint,
        'data' => $postData,
        'retry' => $retryCount,
        'url' => $url
    ]);
    
    try {
        $cookies = loadSmileCookies();
        $cookieStr = buildCookieString($cookies);
        
        $websitePath = '/br';
        foreach ($cookies as $cookie) {
            if (($cookie['name'] ?? '') === 'website_path') {
                $v = urldecode($cookie['value'] ?? '');
                $v = trim($v);
                if ($v === '/br' || strpos($v, '/br/') !== false) { $websitePath = '/br'; break; }
                if ($v === '/ph' || strpos($v, '/ph/') !== false) { $websitePath = '/ph'; break; }
                if ($v === '/id' || strpos($v, '/id/') !== false) { $websitePath = '/id'; break; }
                if ($v === '/mx' || strpos($v, '/mx/') !== false) { $websitePath = '/mx'; break; }
                if ($v === '/th' || strpos($v, '/th/') !== false) { $websitePath = '/th'; break; }
                if ($v === '/us' || strpos($v, '/us/') !== false) { $websitePath = '/us'; break; }
                if ($v === '/pt' || strpos($v, '/pt/') !== false) { $websitePath = '/pt'; break; }
            }
        }
        
        $headers = [
            'User-Agent: Mozilla/5.0 (Windows NT 10.0; Win64; x64) AppleWebKit/537.36 (KHTML, like Gecko) Chrome/120.0.0.0 Safari/537.36',
            'Accept: application/json, text/plain, */*',
            'Accept-Language: en-US,en;q=0.9,pt-BR;q=0.8,pt;q=0.7',
            'Content-Type: application/x-www-form-urlencoded; charset=UTF-8',
            'Cookie: ' . $cookieStr,
            'Origin: ' . SMILE_BASE_URL,
            'Referer: ' . SMILE_BASE_URL . $websitePath . '/mobile-legends',
            'X-Requested-With: XMLHttpRequest',
            'Connection: keep-alive',
            'sec-ch-ua: "Not A(Brand";v="99", "Google Chrome";v="121", "Chromium";v="121"',
            'sec-ch-ua-mobile: ?0',
            'sec-ch-ua-platform: "Windows"',
            'Sec-Fetch-Site: same-origin',
            'Sec-Fetch-Mode: cors',
            'Sec-Fetch-Dest: empty'
        ];
        
        $ch = curl_init($url);
        
        curl_setopt_array($ch, [
            CURLOPT_RETURNTRANSFER => true,
            CURLOPT_POST => true,
            CURLOPT_POSTFIELDS => http_build_query($postData),
            CURLOPT_HTTPHEADER => $headers,
            CURLOPT_HEADER => true,
            CURLOPT_FOLLOWLOCATION => true,
            CURLOPT_MAXREDIRS => 3,
            CURLOPT_TIMEOUT => 40, // Increased timeout for game server
            CURLOPT_SSL_VERIFYHOST => false,
            CURLOPT_SSL_VERIFYPEER => false,
            CURLOPT_ENCODING => 'gzip, deflate, br',
            CURLOPT_HTTP_VERSION => CURL_HTTP_VERSION_1_1,
        ]);
        
        $response = curl_exec($ch);
        $error = curl_error($ch);
        $httpCode = curl_getinfo($ch, CURLINFO_HTTP_CODE);
        
        if ($error) {
            curl_close($ch);
            
            // Retry on connection errors
            if ($retryCount < 3 && strpos($error, 'timeout') !== false) {
                smileLog('Connection timeout, retrying', [
                    'retry' => $retryCount + 1,
                    'error' => $error
                ]);
                sleep(RETRY_DELAY);
                return httpRequest($endpoint, $postData, $retryCount + 1);
            }
            
            throw new Exception('CURL Error: ' . $error);
        }
        
        $headerSize = curl_getinfo($ch, CURLINFO_HEADER_SIZE);
        $headers = substr($response, 0, $headerSize);
        $body = substr($response, $headerSize);
        
        curl_close($ch);
        
        smileLog('HTTP Response', ['status' => $httpCode, 'body_length' => strlen($body), 'retry' => $retryCount]);
        if($retryCount < MAX_RETRIES){
            if($httpCode>=500 || $httpCode==429){
                $wait=RETRY_DELAY*($retryCount+1)+mt_rand(0,2);
                sleep($wait);
                return httpRequest($endpoint,$postData,$retryCount+1);
            }
            $j=@json_decode($body,true);
            if(is_array($j) && (($j['code']??null)===201 || stripos(($j['message']??$j['info']??''),'busy')!==false)){
                $wait=RETRY_DELAY*($retryCount+1)+mt_rand(0,2);
                sleep($wait);
                return httpRequest($endpoint,$postData,$retryCount+1);
            }
            if(!$j && stripos($body,'Game server')!==false){
                $wait=RETRY_DELAY*($retryCount+1)+mt_rand(0,2);
                sleep($wait);
                return httpRequest($endpoint,$postData,$retryCount+1);
            }
        }
        return ['success' => $httpCode == 200 || $httpCode == 302,'status' => $httpCode,'body' => $body,'headers' => $headers];
        
    } catch (Exception $e) {
        smileLog('HTTP Request Exception', ['error' => $e->getMessage()]);
        return [
            'success' => false,
            'error' => $e->getMessage(),
            'status' => 0
        ];
    }
}

// ==================== PURCHASE FUNCTION WITH RETRY ====================
}

function checkSmileBalance() {
    try {
        $cookies = loadSmileCookies();
        $cookieStr = buildCookieString($cookies);
        $ch = curl_init(SMILE_BASE_URL . "/customer/center");
        curl_setopt_array($ch, [
            CURLOPT_RETURNTRANSFER => true,
            CURLOPT_COOKIE => $cookieStr,
            CURLOPT_TIMEOUT => 20,
            CURLOPT_USERAGENT => 'Mozilla/5.0 (Windows NT 10.0; Win64; x64) AppleWebKit/537.36 (KHTML, like Gecko) Chrome/120.0.0.0 Safari/537.36',
            CURLOPT_HEADER => true
        ]);
        $raw = curl_exec($ch);
        $info = curl_getinfo($ch);
        curl_close($ch);
        $hs = $info['header_size'] ?? 0;
        $body = substr($raw ?: '', $hs);
        if (($info['http_code'] ?? 0) !== 200) return ['success'=>false,'error'=>'HTTP '.$info['http_code']];
        if (stripos($body, 'login') !== false) return ['success'=>false,'error'=>'Login required'];
        if (stripos($body, 'Just a moment') !== false) return ['success'=>false,'error'=>'Cloudflare challenge'];
        if (preg_match('/(\d[\\d,\\.]*)\\s*(SC|Coins)/i', $body, $m)) {
            $bal = trim($m[1]).' '.trim($m[2]);
            return ['success'=>true,'balance'=>$bal];
        }
        return ['success'=>true,'balance'=>'Unknown'];
    } catch (Exception $e) {
        return ['success'=>false,'error'=>$e->getMessage()];
    }
}

function processPurchase($userId, $zoneId, $productId) {
    smileLog('=== NEW PURCHASE REQUEST ===', [
        'userId' => $userId,
        'zoneId' => $zoneId,
        'productId' => $productId,
        'time' => time()
    ]);
    
    $pre = checkSmileBalance();
    if (empty($pre['success'])) {
        return ['success'=>false,'error'=>'Cookies invalid or expired'];
    }
    
    // Retry loop for game server errors
    for ($attempt = 1; $attempt <= MAX_RETRIES; $attempt++) {
        smileLog('Purchase attempt ' . $attempt . '/' . MAX_RETRIES);
        
        // STEP 1: QUERY
        $queryResult = httpRequest('/merchant/mobilelegends/query', [
            'user_id' => $userId,
            'zone_id' => $zoneId,
            'pid' => $productId,
            'pay_methond' => 'smilecoin',
            'channel_method' => 'smilecoin'
        ], $attempt - 1);
        
        if (!$queryResult['success']) {
            $errorMsg = 'Query connection failed';
            if (!empty($queryResult['error'])) {
                $errorMsg = $queryResult['error'];
            }
            
            smileLog('Query failed', [
                'attempt' => $attempt,
                'error' => $errorMsg,
                'status' => $queryResult['status'] ?? 0
            ]);
            
            if ($attempt < MAX_RETRIES) {
                $waitTime = RETRY_DELAY * $attempt;
                smileLog('Waiting ' . $waitTime . ' seconds before retry');
                sleep($waitTime);
                continue;
            }
            
            return ['success' => false, 'error' => $errorMsg];
        }
        
        // Parse query response
        $queryData = json_decode($queryResult['body'], true);
        
        if (!is_array($queryData)) {
            smileLog('Invalid query response JSON', [
                'body' => $queryResult['body'],
                'attempt' => $attempt
            ]);
            
            if ($attempt < MAX_RETRIES) {
                sleep(RETRY_DELAY);
                continue;
            }
            
            return ['success' => false, 'error' => 'Invalid server response'];
        }
        
        smileLog('Query response', $queryData);
        
        // Check for game server error (code 201)
        if (isset($queryData['code']) && $queryData['code'] == 201) {
            $errorMsg = $queryData['message'] ?? $queryData['info'] ?? 'Game server error';
            
            smileLog('Game server error detected', [
                'code' => 201,
                'message' => $errorMsg,
                'attempt' => $attempt
            ]);
            
            if ($attempt < MAX_RETRIES) {
                // Progressive wait: 5, 10, 15, 20, 25 seconds
                $waitTime = RETRY_DELAY * $attempt;
                smileLog('Game server busy. Waiting ' . $waitTime . ' seconds');
                sleep($waitTime);
                continue;
            }
            
            return [
                'success' => false,
                'error' => 'Game server is temporarily unavailable. Please try again in 10-15 minutes.'
            ];
        }
        
        // Check for other errors
        if (isset($queryData['code']) && $queryData['code'] != 200) {
            $errorMsg = $queryData['message'] ?? $queryData['info'] ?? 'Query failed';
            
            smileLog('Query error', [
                'code' => $queryData['code'],
                'message' => $errorMsg
            ]);
            
            // Don't retry for these errors
            $noRetryCodes = [400, 401, 403, 404, 422];
            if (in_array($queryData['code'], $noRetryCodes) || 
                stripos($errorMsg, 'invalid') !== false ||
                stripos($errorMsg, 'not found') !== false) {
                
                return ['success' => false, 'error' => $errorMsg];
            }
            
            if ($attempt < MAX_RETRIES) {
                sleep(RETRY_DELAY);
                continue;
            }
            
            return ['success' => false, 'error' => $errorMsg];
        }
        
        // Check for flowid
        if (empty($queryData['flowid'])) {
            smileLog('No flowid in response', ['response' => $queryData]);
            
            if ($attempt < MAX_RETRIES) {
                sleep(RETRY_DELAY);
                continue;
            }
            
            return ['success' => false, 'error' => 'No flowid received'];
        }
        
        $flowId = $queryData['flowid'];
        $nickname = $queryData['nickname'] ?? $queryData['data']['nickname'] ?? 'Unknown';
        
        smileLog('Query successful', [
            'flowId' => $flowId,
            'nickname' => $nickname,
            'attempt' => $attempt
        ]);
        
        // STEP 2: PAYMENT
        $paymentResult = httpRequest('/merchant/mobilelegends/pay', [
            'user_id' => $userId,
            'zone_id' => $zoneId,
            'product_id' => $productId,
            'flowid' => $flowId,
            'pay_methond' => 'smilecoin',
            'channel_method' => 'smilecoin'
        ]);
        
        if (!$paymentResult['success']) {
            $errorMsg = 'Payment failed with HTTP ' . $paymentResult['status'];
            
            smileLog('Payment failed', [
                'status' => $paymentResult['status'],
                'error' => $errorMsg
            ]);
            
            if ($attempt < MAX_RETRIES) {
                sleep(RETRY_DELAY);
                continue;
            }
            
            return [
                'success' => false,
                'error' => $errorMsg,
                'nickname' => $nickname
            ];
        }
        
        // Parse payment response
        $paymentData = json_decode($paymentResult['body'], true);
        if (!is_array($paymentData)) {
            $paymentData = [];
        }
        
        // Check if successful
        if ($paymentResult['status'] == 200 || $paymentResult['status'] == 302) {
            // Extract order ID
            $orderId = null;
            
            // From JSON
            if (isset($paymentData['orderno'])) {
                $orderId = $paymentData['orderno'];
            } elseif (isset($paymentData['orderNo'])) {
                $orderId = $paymentData['orderNo'];
            } elseif (isset($paymentData['data']['orderno'])) {
                $orderId = $paymentData['data']['orderno'];
            }
            
            // From headers (redirect)
            if (!$orderId) {
                $headers = $paymentResult['headers'];
                if (preg_match('/Location:.*[?&](?:order[_-]?(?:id|no))=([^&\s]+)/i', $headers, $matches)) {
                    $orderId = $matches[1];
                }
            }
            
            // Generate if not found
            if (!$orderId) {
                $orderId = 'SMILE_' . date('YmdHis') . '_' . substr(md5($userId . $zoneId), 0, 8);
                smileLog('Generated order ID', ['orderId' => $orderId]);
            }
            
            smileLog('=== PURCHASE SUCCESS ===', [
                'orderId' => $orderId,
                'nickname' => $nickname,
                'attempt' => $attempt,
                'httpStatus' => $paymentResult['status']
            ]);
            
            return [
                'success' => true,
                'order_id' => $orderId,
                'nickname' => $nickname
            ];
        } else {
            // Payment failed
            $errorMsg = 'Payment failed';
            if (isset($paymentData['message'])) {
                $errorMsg = $paymentData['message'];
            }
            
            smileLog('Payment error', [
                'error' => $errorMsg,
                'status' => $paymentResult['status']
            ]);
            
            return [
                'success' => false,
                'error' => $errorMsg,
                'nickname' => $nickname
            ];
        }
    }
    
    return ['success' => false, 'error' => 'All retry attempts failed'];
}

// ==================== API HANDLER ====================

header('Content-Type: application/json; charset=utf-8');
header('Access-Control-Allow-Origin: *');

// Get request data
$input = file_get_contents('php://input');
$data = [];

if (!empty($input)) {
    $json = json_decode($input, true);
    if ($json) {
        $data = $json;
    } else {
        parse_str($input, $data);
    }
}

$data = array_merge($data, $_GET, $_POST);

$action = $data['action'] ?? '';
$userId = $data['user_id'] ?? '';
$zoneId = $data['zone_id'] ?? '';
$productId = $data['product_code'] ?? '';

smileLog('API Request Received', [
    'action' => $action,
    'userId' => $userId,
    'zoneId' => $zoneId,
    'productId' => $productId,
    'ip' => $_SERVER['REMOTE_ADDR'] ?? 'unknown'
]);

// Handle actions
if ($action == 'name-check') {
    if (empty($userId) || empty($zoneId)) {
        echo json_encode(['ok' => 0, 'message' => 'user_id and zone_id required']);
        exit;
    }
    
    $result = processPurchase($userId, $zoneId, '247');
    
    if ($result['success']) {
        echo json_encode([
            'ok' => 1,
            'username' => $result['nickname'],
            'player_id' => $userId,
            'zone_id' => $zoneId
        ], JSON_UNESCAPED_UNICODE);
    } else {
        echo json_encode([
            'ok' => 0,
            'message' => $result['error'] ?? 'Player not found'
        ], JSON_UNESCAPED_UNICODE);
    }
    
} elseif ($action == 'ml_order') {
    if (empty($userId) || empty($zoneId) || empty($productId)) {
        echo json_encode(['ok' => 0, 'message' => 'user_id, zone_id, product_code required']);
        exit;
    }
    
    $result = processPurchase($userId, $zoneId, $productId);
    
    if ($result['success']) {
        echo json_encode([
            'ok' => 1,
            'order_id' => $result['order_id'],
            'username' => $result['nickname'],
            'player_id' => $userId,
            'zone_id' => $zoneId,
            'product_id' => $productId
        ], JSON_UNESCAPED_UNICODE);
    } else {
        // Check if it's game server error
        $errorMsg = $result['error'] ?? 'Order failed';
        
        if (stripos($errorMsg, 'game server') !== false || 
            stripos($errorMsg, 'temporarily unavailable') !== false) {
            
            echo json_encode([
                'ok' => 0,
                'message' => '⚠️ Game server is busy. Please wait 10-15 minutes and try again.',
                'error_type' => 'GAME_SERVER_BUSY',
                'suggestion' => 'Try again after 10-15 minutes'
            ], JSON_UNESCAPED_UNICODE);
        } else {
            echo json_encode([
                'ok' => 0,
                'message' => $errorMsg
            ], JSON_UNESCAPED_UNICODE);
        }
    }
    
} else {
    echo json_encode([
        'ok' => 0,
        'message' => 'SmileOne API. Actions: name-check, ml_order',
        'status' => 'online',
        'timestamp' => time()
    ], JSON_UNESCAPED_UNICODE);
}
?>